package client

import (
	"context"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/emptypb"

	apipb "bdware.org/bdledger/pkg/api/grpc/pb"
)

// Client is a wrapper of apipb.LedgerClient.
// It provides all methods of apipb.LedgerClient but removes
// XXXResponse and returns the actual values.
type Client struct {
	cc *grpc.ClientConn
	nc apipb.NodeClient
	lc apipb.LedgerClient
	qc apipb.QueryClient
	cf Conf
}

// Conf is the configuration of LedgerClient.
// TODO: Add safety certification.
type Conf struct {
	Addr string // address of target gRPC server
}

// Block is one of the parameter of SendTransaction,
// also the gRPC type of the datachain.
type Block = apipb.Block

// Transaction is one of the parameter of SendTransaction.
type Transaction = apipb.SendTransactionRequest_Transaction

// New creates a new LedgerClient.
func New(c Conf) (*Client, error) {
	// connect to server
	con, err := grpc.Dial(c.Addr, grpc.WithInsecure())
	if err != nil {
		return nil, err
	}

	// create clients
	nc := apipb.NewNodeClient(con)
	lc := apipb.NewLedgerClient(con)
	qc := apipb.NewQueryClient(con)

	// all done
	return &Client{
		cc: con,
		nc: nc,
		lc: lc,
		qc: qc,
		cf: c,
	}, nil
}

// Close connection.
func (c *Client) Close() error {
	return c.cc.Close()
}

// Address of client.
func (c *Client) Address() string {
	return c.cf.Addr
}

/////////////////////////////////////////////////////////////////////////////

// ClientVersion gets BDLedger node version
// 查询BDLedger节点版本
func (c *Client) ClientVersion() (string, error) {
	switch res, err := c.nc.ClientVersion(
		context.Background(), &emptypb.Empty{},
	); {
	case err != nil:
		return "", err
	case res == nil:
		return "", status.Error(codes.Internal, "nil response")
	default:
		return res.Version, err
	}
}

// CreateLedger creates a new ledger
// 创建一个新账本
// Return value:
// - true  if ledger created,
// - false if ledger existed.
// - false and error otherwise.
func (c *Client) CreateLedger(name string) (bool, error) {
	switch res, err := c.lc.CreateLedger(
		context.Background(), &apipb.CreateLedgerRequest{
			Name: name,
		},
	); {
	case err != nil:
		return false, err
	case res == nil:
		return false, status.Error(codes.Internal, "nil response")
	default:
		return res.Ok, err
	}
}

// GetLedgers gets all ledgers
// 查询所有帐本列表
func (c *Client) GetLedgers() ([]string, error) {
	switch res, err := c.lc.GetLedgers(
		context.Background(), &emptypb.Empty{},
	); {
	case err != nil:
		return []string{}, err
	case res == nil:
		return []string{}, status.Error(codes.Internal, "nil response")
	default:
		return res.Ledgers, err
	}
}

// SendTransaction sends a new transaction
// 发送一个新事务
// Return value:
// - hash of the Transaction if successful.
// - error if failed.
func (c *Client) SendTransaction(
	ledger string,
	tx Transaction,
) ([]byte, error) {
	switch res, err := c.lc.SendTransaction(
		context.Background(), &apipb.SendTransactionRequest{
			Ledger: ledger,
			Transaction: &apipb.SendTransactionRequest_Transaction{
				Type:  tx.Type,
				From:  tx.From,
				Nonce: tx.Nonce,
				To:    tx.To,
				Data:  tx.Data,
			},
		},
	); {
	case err != nil:
		return []byte{}, err
	case res == nil:
		return []byte{}, status.Error(codes.Internal, "nil response")
	default:
		return res.Hash, err
	}
}

// GetBlockByHash gets a block identified by its hash
// 查询哈希所指定的区块
func (c *Client) GetBlockByHash(
	ledger string,
	hash []byte,
	fullTransactions bool,
) (*Block, error) {
	switch res, err := c.qc.GetBlockByHash(
		context.Background(),
		&apipb.GetBlockByHashRequest{
			Ledger:           ledger,
			Hash:             hash,
			FullTransactions: fullTransactions,
		}); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res.GetBlock(), err
	}
}

// GetBlocks gets blocks in a timestamp range
// 查询时间范围内的区块
// start_timestamp is required
func (c *Client) GetBlocks(
	ledger string,
	filters []*apipb.BlockFilter,
	startTimestamp int64,
	endTimestamp int64,
) (*apipb.GetBlocksResponse, error) {
	switch res, err := c.qc.GetBlocks(
		context.Background(),
		&apipb.BlocksRequest{
			Ledger:         ledger,
			Filters:        filters,
			StartTimestamp: startTimestamp,
			EndTimestamp:   endTimestamp,
		},
	); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res, err
	}
}

// CountBlocks counts all blocks in a ledger, or blocks in a timestamp range
// 查询帐本中的所有区块数量，或时间范围内的区块数量
func (c *Client) CountBlocks(
	ledger string,
	filters []*apipb.BlockFilter,
	startTimestamp int64,
	endTimestamp int64,
) (*apipb.CountBlocksResponse, error) {
	switch res, err := c.qc.CountBlocks(
		context.Background(),
		&apipb.BlocksRequest{
			Ledger:         ledger,
			Filters:        filters,
			StartTimestamp: startTimestamp,
			EndTimestamp:   endTimestamp,
		},
	); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res, err
	}
}

// GetRecentBlocks gets recent 'count' blocks (Only support IncludeTransactions=NONE for now)
// 查询最新的n个区块
func (c *Client) GetRecentBlocks(
	ledger string,
	count int64,
	includeTransactions apipb.IncludeTransactions,
) (*apipb.GetBlocksResponse, error) {
	switch res, err := c.qc.GetRecentBlocks(
		context.Background(),
		&apipb.RecentBlocksRequest{
			Ledger:              ledger,
			Count:               count,
			IncludeTransactions: includeTransactions,
		},
	); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res, err
	}
}

// GetTransactionByHash gets a transaction identified by its hash
// 查询哈希所指定的事务
func (c *Client) GetTransactionByHash(
	ledger string,
	hash []byte,
) (*apipb.Transaction, error) {
	switch res, err := c.qc.GetTransactionByHash(
		context.Background(),
		&apipb.GetTransactionByHashRequest{
			Ledger: ledger,
			Hash:   hash,
		}); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res.GetTransaction(), err
	}
}

// GetTransactionByBlockHashAndIndex gets a transaction identified by hash of the block it belongs to and its index inside the block
// 查询所在区块的哈希与其在区块中的index所指定的事务
func (c *Client) GetTransactionByBlockHashAndIndex(
	ledger string,
	blockHash []byte,
	index uint32,
) (*apipb.Transaction, error) {
	switch res, err := c.qc.GetTransactionByBlockHashAndIndex(
		context.Background(),
		&apipb.GetTransactionByBlockHashAndIndexRequest{
			Ledger:    ledger,
			BlockHash: blockHash,
			Index:     index,
		}); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res.GetTransaction(), err
	}
}

// GetTransactions gets transactions in a timestamp range
// 查询时间范围内的事务
func (c *Client) GetTransactions(
	ledger string,
	startTimestamp int64,
	endTimestamp int64,
	filters []*apipb.TransactionFilter,
) (*apipb.GetTransactionsResponse, error) {
	switch res, err := c.qc.GetTransactions(
		context.Background(),
		&apipb.TransactionsRequest{
			Ledger:         ledger,
			StartTimestamp: startTimestamp,
			EndTimestamp:   endTimestamp,
			Filters:        filters,
		},
	); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res, err
	}
}

// CountTransactions counts all transactions in a ledger, or transactions in a timestamp range
// 查询帐本中的所有事务数量，或时间范围内的事务数量
// start_timestamp is required
func (c *Client) CountTransactions(
	ledger string,
	startTimestamp int64,
	endTimestamp int64,
	filters []*apipb.TransactionFilter,
) (*apipb.CountTransactionsResponse, error) {
	switch res, err := c.qc.CountTransactions(
		context.Background(),
		&apipb.TransactionsRequest{
			Ledger:         ledger,
			StartTimestamp: startTimestamp,
			EndTimestamp:   endTimestamp,
			Filters:        filters,
		},
	); {
	case err != nil:
		return nil, err
	case res == nil:
		return nil, status.Error(codes.Internal, "nil response")
	default:
		return res, err
	}
}
